/*
* Copyright (c) 2015 SODAQ. All rights reserved.
*
* This file is part of Sodaq_RN2483.
*
* Sodaq_RN2483 is free software: you can redistribute it and/or modify
* it under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation, either version 3 of
* the License, or(at your option) any later version.
*
* Sodaq_RN2483 is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with Sodaq_RN2483.  If not, see
* <http://www.gnu.org/licenses/>.
*/

#include <Sodaq_RN2483.h>
#include <OneWire.h>
#include <DallasTemperature.h>

#define debugSerial SerialUSB
#define loraSerial Serial1
#define beePin BEE_VCC

//const uint8_t devAddr[4] = { };
//const uint8_t appSKey[16] = { };
//const uint8_t nwkSKey[16] = { };

// USE YOUR OWN KEYS!
const uint8_t devEUI[8] =
{ };

// USE YOUR OWN KEYS!
const uint8_t appEUI[8] =
{ };

const uint8_t appKey[16] =
{ };

//Analog pins
//A1 may be battery voltage pin
int light_pin = A0;
int moisture_pin = A2; 
//int battery_volt_pin = A1;
//Digital pins
int temperature_pin = 0;
int temperature_vcc_pin = 1;
int moisture_vcc_pin = 8;
int moisture_gnd_pin = 7;

//constants for reading battery voltage
#define BATVOLTPIN  BAT_VOLT
static const double ADC_AREF = 3.3;
static const double BATVOLT_R1 = 4.7;
static const double BATVOLT_R2 = 10;

OneWire oneWire(temperature_pin);
// Pass our oneWire reference to Dallas Temperature.
DallasTemperature sensors(&oneWire);

void setup()
{
  // Start up the temperature sensor library
  pinMode(temperature_vcc_pin, OUTPUT);
  digitalWrite(temperature_vcc_pin,HIGH);
  pinMode(moisture_vcc_pin, OUTPUT);
  digitalWrite(moisture_vcc_pin,HIGH);
  pinMode(moisture_gnd_pin, OUTPUT);
  digitalWrite(moisture_gnd_pin,LOW);
  sensors.begin();
    
  debugSerial.begin(57600);

  loraSerial.begin(LoRaBee.getDefaultBaudRate());

  digitalWrite(beePin, HIGH);
  pinMode(beePin, OUTPUT);
  
  pinMode(LED_BUILTIN, OUTPUT);
  setupNetwork();
}

void setupNetwork() {  
  LoRaBee.setDiag(debugSerial); // optional
  //if(LoRaBee.initABP(loraSerial, devAddr, appSKey, nwkSKey, false))
  if (LoRaBee.initOTA(loraSerial, devEUI, appEUI, appKey, true))
  {
    debugSerial.println("Connection to the network was successful.");
  }
  else
  {
    debugSerial.println("Connection to the network failed!");
  }

  LoRaBee.configChFreq(0, 868100000L,0,5,1);
  LoRaBee.configChFreq(1, 868300000L,0,5,1); 
  LoRaBee.configChFreq(2, 868500000L,0,5,1);
  LoRaBee.configChFreq(3, 867100000L,0,5,1);
  LoRaBee.configChFreq(4, 867300000L,0,5,1);
  LoRaBee.configChFreq(5, 867500000L,0,5,1);
  LoRaBee.configChFreq(6, 867700000L,0,5,1);
  LoRaBee.configChFreq(7, 867900000L,0,5,1);
}
  
// LMIC code limits maximum payload size to 52 octets (see lmic.h)
// This corresponds to datarate 0 (SF12) in specification 1.0 => worst case
String message;

void loop()
{
  debugSerial.println("Sending out test packets.");
  for (uint8_t i = 5; i > 0; i--)
  {
    debugSerial.println(i);
    delay(1000);
  }
  
  sensors.requestTemperatures(); // Send the command to get temperatures
  // Verify encoding
  //message = "{\"L\": " + String(analogRead(light_pin)) +", " +"\"M\": " + String(analogRead(moisture_pin))+", " +"\"T\": " + String(sensors.getTempCByIndex(0)) + ", "+"\"B\": " + String(getRealBatteryVoltageMV())  +"}";
  message = "hello";
    
  debugSerial.println("Sending: "+message);
  
  switch (LoRaBee.send(1, (uint8_t*)message.c_str(), message.length()))
  {
  case NoError:
    debugSerial.println("Successful transmission.");
    digitalWrite(LED_BUILTIN, HIGH);
    break;
  case NoResponse:
    debugSerial.println("There was no response from the device.");
    digitalWrite(LED_BUILTIN, LOW);
    break;
  case Timeout:
    debugSerial.println("Connection timed-out. Check your serial connection to the device! Sleeping for 20sec.");
    digitalWrite(LED_BUILTIN, LOW);
    delay(20000);
    break;
  case PayloadSizeError:
    debugSerial.println("The size of the payload is greater than allowed. Transmission failed!");
    digitalWrite(LED_BUILTIN, LOW);
    break;
  case InternalError:
    debugSerial.println("Oh No! This shouldn't happen. Something is really wrong! Try restarting the device!\r\nThe program will now halt.");
    setupNetwork();
    digitalWrite(LED_BUILTIN, LOW);
    break;
  case Busy:
    debugSerial.println("The device is busy. Sleeping for 10 extra seconds.");
    blink();
    delay(20000);
    break;
  case NoFreeChannel:
    debugSerial.println("The device has no free channel. Sleeping for 10 extra seconds.");
    blink();
    delay(10000);
    break;
  case Silent:
    debugSerial.println("The device is silent. Sleeping for 10 extra seconds.");
    blink();
    delay(20000);
    break;
  case NetworkFatalError:
    debugSerial.println("There is a non-recoverable error with the network connection. You should re-connect.\r\nThe program will now halt.");
    digitalWrite(LED_BUILTIN, LOW);
    setupNetwork();
    break;
  case NotConnected:
    debugSerial.println("The device is not connected to the network. Please connect to the network before attempting to send data.\r\nThe program will now halt.");
    digitalWrite(LED_BUILTIN, LOW);
    setupNetwork();
    break;
  case NoAcknowledgment:
    debugSerial.println("There was no acknowledgment sent back!");
    digitalWrite(LED_BUILTIN, LOW);
    break;
  default:
    break;
  }
  delay(80000);
}

void blink() {
  for (uint8_t i = 10; i > 0; i--)
  {
    digitalWrite(LED_BUILTIN, HIGH);   // turn the LED on (HIGH is the voltage level)
    delay(1000);                       // wait for a second
    digitalWrite(LED_BUILTIN, LOW);    // turn the LED off by making the voltage LOW
    delay(1000);                       // wait for a second
  }
}

double getRealBatteryVoltageMV()
{
  uint16_t batteryVoltage = analogRead(BATVOLTPIN);
  return (ADC_AREF / 1.023) * (BATVOLT_R1 + BATVOLT_R2) / BATVOLT_R2 * batteryVoltage;
}
